/***************************************************************************

  timer.c

  Functions needed to generate timing and synchronization between several
  CPUs.

  Changes 2/27/99:
  	- added some rounding to the sorting of timers so that two timers
  		allocated to go off at the same time will go off in the order
  		they were allocated, without concern for floating point rounding
  		errors (thanks Juergen!)
  	- fixed a bug where the base_time was not updated when a CPU was
  		suspended, making subsequent calls to getabsolutetime() return an
  		incorrect time (thanks Nicola!)
  	- changed suspended CPUs so that they don't eat their timeslice until
  		all other CPUs have used up theirs; this allows a slave CPU to
  		trigger a higher priority CPU in the middle of the timeslice
  	- added the ability to call timer_reset() on a oneshot or pulse timer
  		from within that timer's callback; in this case, the timer won't
  		get removed (oneshot) or won't get reprimed (pulse)

  Changes 12/17/99 (HJB):
	- added overclocking factor and functions to set/get it at runtime.

  Changes 12/23/99 (HJB):
	- added burn() function pointer to tell CPU cores when we want to
	  burn cycles, because the cores might need to adjust internal
	  counters or timers.

***************************************************************************/

#include "mamewrap.h"
#include "bochs.h"
extern "C" {
#include "timer.h"
}


#define VERBOSE 0

#define MAX_TIMERS 256


/*
 *		internal timer structures
 */
typedef struct timer_entry
{
	struct timer_entry *next;
	struct timer_entry *prev;
	void (*callback)(int);
	int callback_param;
	int enabled;
	double period;
	double start;
	double expire;
} timer_entry;

/* list of active timers */
static timer_entry timers[MAX_TIMERS];
static timer_entry *timer_head;
static timer_entry *timer_free_head;

/* other internal states */
static double start_time;
static double global_offset;
static timer_entry *callback_timer;
static int callback_timer_modified;
static int timer_handle;

#if VERBOSE
static void verbose_print(char *s, ...);
#endif

/*
 *		return the current absolute time
 */
INLINE double getabsolutetime(void)
{
  return(bx_pc_system.get_current_time()/1000000.0-start_time);
}


/*
 *		allocate a new timer
 */
INLINE timer_entry *timer_new(void)
{
	timer_entry *timer;

	/* remove an empty entry */
	if (!timer_free_head)
		return NULL;
	timer = timer_free_head;
	timer_free_head = timer->next;

	return timer;
}


/*
 *		insert a new timer into the list at the appropriate location
 */
INLINE void timer_list_insert(timer_entry *timer)
{
	double expire = timer->enabled ? timer->expire : TIME_NEVER;
	timer_entry *t, *lt = NULL;

	/* loop over the timer list */
	for (t = timer_head; t; lt = t, t = t->next)
	{
		/* if the current list entry expires after us, we should be inserted before it */
		/* note that due to floating point rounding, we need to allow a bit of slop here */
		/* because two equal entries -- within rounding precision -- need to sort in */
		/* the order they were inserted into the list */
		if ((t->expire - expire) > TIME_IN_NSEC(1))
		{
			/* link the new guy in before the current list entry */
			timer->prev = t->prev;
			timer->next = t;

			if (t->prev)
				t->prev->next = timer;
			else
				timer_head = timer;
			t->prev = timer;
			return;
		}
	}

	/* need to insert after the last one */
	if (lt)
		lt->next = timer;
	else
		timer_head = timer;
	timer->prev = lt;
	timer->next = NULL;
}


/*
 *		remove a timer from the linked list
 */
INLINE void timer_list_remove(timer_entry *timer)
{
	/* remove it from the list */
	if (timer->prev)
		timer->prev->next = timer->next;
	else
		timer_head = timer->next;
	if (timer->next)
		timer->next->prev = timer->prev;
}


/*
 *		initialize the timer system
 */
void timer_init(void)
{
	int i;

	start_time = 0;
	start_time = getabsolutetime();

	/* we need to wait until the first call to timer_cyclestorun before using real CPU times */
	global_offset = 0.0;
	callback_timer = NULL;
	callback_timer_modified = 0;

	/* reset the timers */
	memset(timers, 0, sizeof(timers));

	/* initialize the lists */
	timer_head = NULL;
	timer_free_head = &timers[0];
	for (i = 0; i < MAX_TIMERS-1; i++)
		timers[i].next = &timers[i+1];

	timer_handle = bx_pc_system.register_timer((void*)timer_handler,
						   timer_handler,
						   0,
						   0,
						   0);
}

/*
 *		allocate a pulse timer, which repeatedly calls the callback using the given period
 */
void *timer_pulse(double period, int param, void (*callback)(int))
{
	double time = getabsolutetime();
	timer_entry *timer;

	/* allocate a new entry */
	timer = timer_new();
	if (!timer)
		return NULL;

	/* fill in the record */
	timer->callback = callback;
	timer->callback_param = param;
	timer->enabled = 1;
	timer->period = period;

	/* compute the time of the next firing and insert into the list */
	timer->start = time;
	timer->expire = time + period;
	timer_list_insert(timer);

	timer_update();

	#if VERBOSE
		verbose_print("T=%.6g: New pulse=%08X, period=%.6g\n", time + global_offset, timer, period);
	#endif

	/* return a handle */
	return timer;
}


/*
 *		allocate a one-shot timer, which calls the callback after the given duration
 */
void *timer_set(double duration, int param, void (*callback)(int))
{
	double time = getabsolutetime();
	timer_entry *timer;

	/* allocate a new entry */
	timer = timer_new();
	if (!timer)
		return NULL;

	/* fill in the record */
	timer->callback = callback;
	timer->callback_param = param;
	timer->enabled = 1;
	timer->period = 0;

	/* compute the time of the next firing and insert into the list */
	timer->start = time;
	timer->expire = time + duration;
	timer_list_insert(timer);

	timer_update();

	#if VERBOSE
		verbose_print("T=%.6g: New oneshot=%08X, duration=%.6g\n", time + global_offset, timer, duration);
	#endif

	/* return a handle */
	return timer;
}


/*
 *		reset the timing on a timer
 */
void timer_reset(void *which, double duration)
{
	double time = getabsolutetime();
	timer_entry *timer = (timer_entry *)which;

	/* compute the time of the next firing */
	timer->start = time;
	timer->expire = time + duration;

	/* remove the timer and insert back into the list */
	timer_list_remove(timer);
	timer_list_insert(timer);

	timer_update();

	/* if this is the callback timer, mark it modified */
	if (timer == callback_timer)
		callback_timer_modified = 1;

	#if VERBOSE
		verbose_print("T=%.6g: Reset %08X, duration=%.6g\n", time + global_offset, timer, duration);
	#endif
}


/*
 *		remove a timer from the system
 */
void timer_remove(void *which)
{
	timer_entry *timer = (timer_entry *)which;

	/* remove it from the list */
	timer_list_remove(timer);

	/* free it up by adding it back to the free list */
	timer->next = timer_free_head;
	timer_free_head = timer;

	#if VERBOSE
		verbose_print("T=%.6g: Removed %08X\n", getabsolutetime() + global_offset, timer);
	#endif
}


/*
 *		enable/disable a timer
 */
int timer_enable(void *which, int enable)
{
	timer_entry *timer = (timer_entry *)which;
	int old;

	#if VERBOSE
		if (enable) verbose_print("T=%.6g: Enabled %08X\n", getabsolutetime() + global_offset, timer);
		else verbose_print("T=%.6g: Disabled %08X\n", getabsolutetime() + global_offset, timer);
	#endif

	/* set the enable flag */
	old = timer->enabled;
	timer->enabled = enable;

	/* remove the timer and insert back into the list */
	timer_list_remove(timer);
	timer_list_insert(timer);

	return old;
}


/*
 *		return the time since the last trigger
 */
double timer_timeelapsed(void *which)
{
	double time = getabsolutetime();
	timer_entry *timer = (timer_entry *)which;

	return time - timer->start;
}


/*
 *		return the time until the next trigger
 */
double timer_timeleft(void *which)
{
	double time = getabsolutetime();
	timer_entry *timer = (timer_entry *)which;

	return timer->expire - time;
}


/*
 *		return the current time
 */
double timer_get_time(void)
{
	return global_offset + getabsolutetime();
}


/*
 *		return the time when this timer started counting
 */
double timer_starttime(void *which)
{
	timer_entry *timer = (timer_entry *)which;
	return global_offset + timer->start;
}


/*
 *		return the time when this timer will fire next
 */
double timer_firetime(void *which)
{
	timer_entry *timer = (timer_entry *)which;
	return global_offset + timer->expire;
}


/*
 *		begin CPU execution by determining how many cycles the CPU should run
 */
void timer_handler(void *this_ptr)
{
	while (timer_head != NULL &&
	       timer_head->expire <= getabsolutetime())
	{
		timer_entry *timer = timer_head;

		#if VERBOSE
			verbose_print("T=%.6g: %08X fired (exp time=%.6g)\n", getabsolutetime() + global_offset, timer, timer->expire + global_offset);
		#endif

		/* set the global state of which callback we're in */
		callback_timer_modified = 0;
		callback_timer = timer;

		/* call the callback */
		if (timer->callback)
		{
			profiler_mark(PROFILER_TIMER_CALLBACK);
			(*timer->callback)(timer->callback_param);
			profiler_mark(PROFILER_END);
		}

		/* clear the callback timer global */
		callback_timer = NULL;

		/* reset or remove the timer, but only if it wasn't modified during the callback */
		if (!callback_timer_modified)
		{
			if (timer->period)
			{
				timer->start = timer->expire;
				timer->expire += timer->period;

				timer_list_remove(timer);
				timer_list_insert(timer);
			}
			else
				timer_remove(timer);
		}
	}

	timer_update();
}

void timer_update(void)
{
  double newbeat;
  int newbeat2;

  newbeat = 10000000000000.0;
  if(timer_head != NULL) {
    newbeat = (timer_head->expire - getabsolutetime()) * 1000000.0;
  }
  if(newbeat<0) {
    newbeat2=0;
  } else if(newbeat>1000000) {
    newbeat2=1000000;
  } else {
    newbeat2=(int)newbeat;
  }
  if(newbeat2<1) {
    newbeat2=1;
  }
  if(newbeat2>100000) {
    newbeat2=100000;
  }
  bx_pc_system.activate_timer(timer_handle,
			      newbeat2,
			      0);
}


/*
 *		debugging
 */
#if VERBOSE

#ifdef macintosh
#undef printf
#endif

static void verbose_print(char *s, ...)
{
	va_list ap;

	va_start(ap, s);

	#if (VERBOSE == 1)
		if (errorlog) vfprintf(errorlog, s, ap);
	#else
		vprintf(s, ap);
		fflush(NULL);
	#endif

	va_end(ap);
}

#endif
